#include <Wire.h>
#include <SSD1306Ascii.h>
#include <SSD1306AsciiWire.h>
#include <DHT.h>
#include <IRremote.hpp>
#include <Adafruit_NeoPixel.h>
#include <ESP32Servo.h>  // Servo-Bibliothek

// Pin-Konfiguration
#define LED_RED 25
#define LED_GREEN 26
#define BUTTON_PIN 23
#define POTENTIOMETER_PIN 39
#define DHTPIN 14
#define DHTTYPE DHT11
#define WS2812_PIN 15
#define NUM_LEDS 8
#define FAN_PIN 4
#define SERVO_PIN 12
#define MQ135_PIN 34
#define SOUND_PIN 19
#define PIR_SENSOR_PIN 13
#define LDR_PIN 35
#define IR_RECEIVER_PIN 16  // Pin für IR-Empfänger

// Objekte initialisieren
SSD1306AsciiWire oled;
DHT dht(DHTPIN, DHTTYPE);                    // DHT-Sensor initialisieren
IRrecv irReceiver(IR_RECEIVER_PIN);           // IR-Empfänger-Objekt
Adafruit_NeoPixel strip(NUM_LEDS, WS2812_PIN, NEO_GRB + NEO_KHZ800);  // WS2812 LEDs
Servo servo;                                  // Servo-Objekt
decode_results results;                       // Struktur für die IR-Daten

// Menüeinträge
const char* menuItems[] = {
  "Rote LED",
  "Gruene LED",
  "DHT11 auslesen",
  "WS2812 LEDs",
  "Servo bewegen",
  "Luefter starten",
  "MQ135 auslesen",
  "Lautsprecher testen",
  "PIR-Sensor testen",
  "LDR auslesen",
  "IR-Empfaenger testen"
};
const int menuSize = sizeof(menuItems) / sizeof(menuItems[0]);

int selectedMenu = 0;
int currentPage = 0;
int previousPage = -1;
bool isModuleActive = false;

void setup() {
  Wire.begin();
  oled.begin(&Adafruit128x64, 0x3C);
  oled.setFont(System5x7);
  oled.clear();

  pinMode(LED_RED, OUTPUT);
  pinMode(LED_GREEN, OUTPUT);
  pinMode(FAN_PIN, OUTPUT);
  pinMode(BUTTON_PIN, INPUT);
  pinMode(PIR_SENSOR_PIN, INPUT);
  pinMode(LDR_PIN, INPUT);

  // Initialisierung
  dht.begin();
  irReceiver.begin(IR_RECEIVER_PIN, ENABLE_LED_FEEDBACK);
  strip.begin();
  strip.show();  // Alle LEDs ausschalten
  servo.attach(SERVO_PIN);

  oled.println("Funduino Auswahl");
  delay(2000);
}

void loop() {
  if (!isModuleActive) {
    int potValue = analogRead(POTENTIOMETER_PIN);
    selectedMenu = map(potValue, 0, 4095, 0, menuSize - 1);

    currentPage = selectedMenu / 4;

    if (currentPage != previousPage) {
      oled.clear();
      previousPage = currentPage;
    }

    displayMenu();

    if (digitalRead(BUTTON_PIN) == HIGH) {
      delay(200);
      activateModule(selectedMenu);
    }
  }
}

// Menü anzeigen
void displayMenu() {
  oled.setCursor(0, 0);
  oled.println("Modul-Auswahl:");

  int startItem = currentPage * 4;
  int endItem = min(startItem + 4, menuSize);

  for (int i = startItem; i < endItem; i++) {
    oled.setCursor(0, i - startItem + 1);
    oled.print("                    ");
    oled.setCursor(0, i - startItem + 1);

    if (i == selectedMenu) {
      oled.print("> ");
    } else {
      oled.print("  ");
    }
    oled.println(menuItems[i]);
  }
}

// Modul aktivieren
void activateModule(int menuIndex) {
  isModuleActive = true;
  oled.clear();
  oled.setCursor(0, 0);
  oled.println("Aktiviere:");
  oled.println(menuItems[menuIndex]);

  switch (menuIndex) {
    case 0:  // Rote LED
      digitalWrite(LED_RED, HIGH);
      delay(6000);
      digitalWrite(LED_RED, LOW);
      break;

    case 1:  // Gruene LED
      digitalWrite(LED_GREEN, HIGH);
      delay(6000);
      digitalWrite(LED_GREEN, LOW);
      break;

    case 2:  // DHT11 auslesen
      for (int i = 0; i < 6; i++) {
        float temp = dht.readTemperature();
        float hum = dht.readHumidity();
        oled.setCursor(0, 2);
        oled.print("Temp: ");
        oled.print(temp);
        oled.println(" C");
        oled.setCursor(0, 3);
        oled.print("Feuchte: ");
        oled.print(hum);
        oled.println(" %");
        delay(1000);
      }
      break;

    case 3:  // WS2812 LEDs
      for (int i = 0; i < 6; i++) {
        for (int j = 0; j < NUM_LEDS; j++) {
          strip.setPixelColor(j, strip.Color(random(0, 255), random(0, 255), random(0, 255)));
        }
        strip.show();
        delay(1000);
      }
      strip.clear();
      strip.show();
      break;

    case 4:  // Servo bewegen
      for (int i = 0; i < 3; i++) {
        servo.write(180);
        delay(1000);
        servo.write(90);
        delay(1000);
        servo.write(0);
        delay(1000);
        servo.write(90);
        delay(1000);
      }
      servo.write(90);
      break;

    case 5:  // Lüfter starten
      digitalWrite(FAN_PIN, HIGH);
      delay(6000);
      digitalWrite(FAN_PIN, LOW);
      break;

    case 6:  // MQ135 auslesen
      for (int i = 0; i < 6; i++) {
        int airValue = analogRead(MQ135_PIN);
        oled.setCursor(0, 2);
        oled.print("MQ135-Wert: ");
        oled.println(airValue);
        delay(1000);
      }
      break;

    case 7:  // Lautsprecher testen
      testSpeaker();
      break;

    case 8:  // PIR-Sensor testen (digitaler Wert)
      for (int i = 0; i < 5; i++) {
        oled.setCursor(0, 2);
        oled.print("PIR Status:");
        oled.setCursor(0, 3);
        int pirValue = digitalRead(PIR_SENSOR_PIN);
        oled.print(pirValue == HIGH ? "Bewegung erkannt" : "Keine Bewegung");
        delay(1000);
      }
      break;

    case 9:  // LDR auslesen
      for (int i = 0; i < 6; i++) {
        int lightValue = analogRead(LDR_PIN);
        oled.setCursor(0, 2);
        oled.print("LDR-Wert: ");
        oled.println(lightValue);
        delay(1000);
      }
      break;

    case 10:  // IR-Empfänger testen
      for (int i = 0; i < 10; i++) {
        oled.setCursor(0, 2);
        oled.print("IR-Code:");

        if (irReceiver.decode()) {
          oled.setCursor(0, 3);
          oled.print("0x");
          oled.println(irReceiver.decodedIRData.decodedRawData, HEX);

          // Ausgabe über die serielle Konsole
          irReceiver.printIRResultShort(&Serial);

          irReceiver.resume();  // Nächstes Signal empfangen
        } else {
          oled.setCursor(0, 3);
          oled.print("Warten...");
        }
        delay(1000);
      }
      break;
  }

  isModuleActive = false;
}

// Funktion für den Lautsprecher-Test
void testSpeaker() {
  ledcAttach(SOUND_PIN, NOTE_A, 7);
  delay(115);
  ledcWriteNote(SOUND_PIN, NOTE_A, 7);
  delay(900);
  ledcWrite(SOUND_PIN, 0);
  ledcDetach(SOUND_PIN);
}
